<?php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Headers: Content-Type");
header("Access-Control-Allow-Methods: POST");
header('Content-Type: application/json');

// Only accept POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(["status" => "error", "message" => "Method Not Allowed"]);
    exit;
}

// Get and decode JSON input
$input = file_get_contents("php://input");
$data = json_decode($input, true);

// Validate required fields
if (empty($data['name']) || empty($data['phone'])) {
    http_response_code(400);
    echo json_encode(["status" => "error", "message" => "Missing required fields."]);
    exit;
}

// Sanitize input
$name = htmlspecialchars(trim($data['name']));
$phone = preg_replace('/\D/', '', $data['phone']); // Only digits
$age = isset($data['age']) ? htmlspecialchars(trim($data['age'])) : '';
$user_weight = isset($data['user_weight']) ? htmlspecialchars(trim($data['user_weight'])) : '';

// Handle user_goal as comma-separated string
$user_goal_array = $data['user_goal'] ?? [];
if (is_array($user_goal_array)) {
    $sanitized_goals = array_map(function ($goal) {
        return htmlspecialchars(trim($goal));
    }, $user_goal_array);
    $user_goal = implode(', ', $sanitized_goals);
} else {
    $user_goal = htmlspecialchars(trim($user_goal_array));
}

// Validations
if (!preg_match("/^\d{10}$/", $phone)) {
    http_response_code(400);
    echo json_encode(["status" => "error", "message" => "Invalid phone number format."]);
    exit;
}

if (strlen($name) < 3) {
    http_response_code(400);
    echo json_encode(["status" => "error", "message" => "Name must be at least 3 characters long."]);
    exit;
}

// Prepare payload
$orderid = mt_rand(10000, 99999);
$orderdate = date('Y-m-d');

$payload = [
    'orderid'           => $orderid,
    'fullname'          => $name,
    'mobile'            => $phone,
    'age'               => $age,
    'weight'            => $user_weight,
    'weightlossgoal'    => $user_goal,
    'orderdate'         => $orderdate
];
// Log request
$logData = "[" . date('Y-m-d H:i:s') . "] REQUEST:\n" . print_r($payload, true) . "\n";
file_put_contents('form-log.txt', $logData, FILE_APPEND);

// Send data via cURL
$curl = curl_init();
curl_setopt_array($curl, [
    CURLOPT_URL => 'http://shrichitt.in/crm/api/weight-loss',
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_ENCODING => '',
    CURLOPT_MAXREDIRS => 10,
    CURLOPT_TIMEOUT => 10,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
    CURLOPT_CUSTOMREQUEST => 'POST',
    CURLOPT_POSTFIELDS => json_encode($payload),
    CURLOPT_HTTPHEADER => [
        'Content-Type: application/json'
    ],
]);

$response = curl_exec($curl);
$curl_error = curl_error($curl);
curl_close($curl);

if ($curl_error) {
    http_response_code(500);
    echo json_encode(["status" => "error", "message" => "CURL Error: " . $curl_error]);
} else {
    $decoded_response = json_decode($response, true);
    $payload['api_response'] = $decoded_response ?: $response;

    // Log success with response
    $successLog = "[" . date('Y-m-d H:i:s') . "] SUCCESS:\n" . print_r($payload, true) . "\n";
    file_put_contents('success-form-log.txt', $successLog, FILE_APPEND);

    echo json_encode(["status" => "success", "message" => "Data Inserted Successfully"]);
}
